from flask import Flask, render_template, request, redirect, url_for, abort
from fsfpay import FsfPay
import os

app = Flask(__name__)

# Load credentials from env or set defaults (replace with real secrets)
FSF_USERNAME = os.getenv('FSFPAY_USERNAME', 'YOUR_FSFPAY_USERNAME')
FSF_APIKEY   = os.getenv('FSFPAY_API_KEY', 'YOUR_FSFPAY_API_KEY')
FSF_SECRET   = os.getenv('FSFPAY_SECRET', 'YOUR_FSFPAY_SECRET')

# Real payments by default (testmode=0)
fsfpay = FsfPay(api_username=FSF_USERNAME, api_key=FSF_APIKEY, api_secret=FSF_SECRET, testmode=0)

@app.route('/', methods=['GET','POST'])
def index():
    error = None
    try:
        coins = fsfpay.get_cryptocurrencies()
    except Exception as e:
        coins = {}
        error = str(e)

    if request.method == 'POST':
        coin = request.form.get('coin')
        order_id = request.form.get('order_id') or 'order-' + str(os.urandom(4).hex())
        email = request.form.get('email') or 'customer@example.com'
        amount = request.form.get('amount') or '10.00'
        currency = request.form.get('currency') or 'USD'
        callback_url = request.form.get('callback') or url_for('callback', _external=True)
        success_url = request.form.get('success') or url_for('success', _external=True)

        payment = fsfpay.create_payment(
            order_id=order_id,
            email=email,
            amount=amount,
            currency=currency,
            cryptocurrency=coin,
            callback_url=callback_url,
            success_url=success_url,
            language='en'
        )

        if payment.get('success'):
            return render_template('iframe.html', iframe_url=payment.get('iframe_url'))
        else:
            return f"Payment Error: {payment.get('message')}<br><pre>{payment.get('raw')}</pre>", 400

    return render_template('index.html', coins=coins, error=error)

@app.route('/callback', methods=['POST'])
def callback():
    # Try to read form or json
    payload = request.form.to_dict() or request.get_json(silent=True) or {}
    # Verify callback authenticity
    if not FsfPay.verify_callback_payload(payload, FSF_APIKEY, FSF_USERNAME, FSF_SECRET):
        # log warning
        app.logger.warning('Invalid FSFPAY callback signature: %s', payload)
        return 'Invalid signature', 400

    # process callback (update DB etc.)
    app.logger.info('FSFPAY callback verified: %s', payload)
    # Example: check payload.get('status') and update order in DB
    return 'OK', 200

@app.route('/success')
def success():
    return 'Payment succeeded. Thank you.'

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=True)
