import requests
import hashlib

class FsfPay:
    def __init__(self, api_username, api_key, api_secret, base_url="https://panel.fsfpay.com/api", testmode=0):
        self.api_username = api_username
        self.api_key = api_key
        self.api_secret = api_secret
        self.base_url = base_url.rstrip('/')
        self.testmode = int(testmode)

    def _generate_hash(self):
        return hashlib.sha256((self.api_key + self.api_username + self.api_secret).encode('utf-8')).hexdigest()

    def get_cryptocurrencies(self):
        data = {
            'apiKey': self.api_key,
            'apiUsername': self.api_username,
            'requestHash': self._generate_hash()
        }
        res = requests.post(f"{self.base_url}/cryptocurrencies", data=data, timeout=30)
        res.raise_for_status()
        return res.json().get('cryptocurrencies', {})

    def create_payment(self, order_id, email, amount, currency, cryptocurrency, callback_url, success_url, language='en'):
        data = {
            'apiKey': self.api_key,
            'apiUsername': self.api_username,
            'orderId': order_id,
            'email': email,
            'amount': amount,
            'currency': currency,
            'cryptocurrency': cryptocurrency,
            'callbackUrl': callback_url,
            'successRedirectUrl': success_url,
            'language': language,
            'testmode': self.testmode,
            'requestHash': self._generate_hash()
        }
        res = requests.post(f"{self.base_url}/payment", data=data, timeout=30)
        res.raise_for_status()
        response = res.json()

        # FSFPAY may return iframeUrl or paymentUrl depending on implementation
        iframe_url = response.get('iframeUrl') or response.get('paymentUrl') or response.get('redirectUrl')
        if iframe_url:
            return {
                'success': True,
                'iframe_url': iframe_url,
                'payment_id': response.get('paymentId'),
                'script_url': response.get('scriptUrl'),
                'raw': response
            }
        return {
            'success': False,
            'message': response.get('message', 'Unknown error'),
            'raw': response
        }

    @staticmethod
    def verify_callback_payload(payload, api_key, api_username, api_secret):
        """Verify callback using requestHash field (SHA256 of apiKey+apiUsername+apiSecret)."""
        expected = hashlib.sha256((api_key + api_username + api_secret).encode('utf-8')).hexdigest()
        # payload may be dict-like with form fields
        received = payload.get('requestHash') or payload.get('request_hash') or payload.get('requesthash')
        return received == expected
