<?php
/**
 * Plugin Name: FSFPAY Payment Gateway
 * Plugin URI:  https://panel.fsfpay.com/
 * Description: FSFPAY WooCommerce gateway — hosted checkout + webhook. Configure API keys & endpoints in WooCommerce > Settings > Payments > FSFPAY.
 * Version:     1.0.0
 * Author:      FSFPAY Integration
 * Text Domain: fsfpay-payment-gateway
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

add_action( 'plugins_loaded', 'fsfpay_init_gateway', 11 );
function fsfpay_init_gateway() {
    if ( ! class_exists( 'WC_Payment_Gateway' ) ) return;

    class WC_Gateway_Fsfpay extends WC_Payment_Gateway {
        public function __construct() {
            $this->id                 = 'fsfpay';
            $this->icon               = ''; 
            $this->has_fields         = false; 
            $this->method_title       = __( 'FSFPAY', 'fsfpay-payment-gateway' );
            $this->method_description = __( 'FSFPAY payment gateway', 'fsfpay-payment-gateway' );

            $this->init_form_fields();
            $this->init_settings();

            $this->title        = $this->get_option( 'title' );
            $this->description  = $this->get_option( 'description' );
            $this->enabled      = $this->get_option( 'enabled' );
            $this->api_key      = $this->get_option( 'api_key' );
            $this->api_secret   = $this->get_option( 'api_secret' );
            $this->test_mode    = 'yes' === $this->get_option( 'test_mode' );
            $this->test_endpoint = $this->get_option( 'test_endpoint', 'https://panel.fsfpay.com/api/sandbox/checkout' );
            $this->live_endpoint = $this->get_option( 'live_endpoint', 'https://panel.fsfpay.com/api/checkout' );

            add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
            add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'receipt_page' ) );
            add_action( 'rest_api_init', array( $this, 'register_webhook_route' ) );

            $this->logger = wc_get_logger();
            $this->log_channel = 'fsfpay';
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title'   => __( 'Enable/Disable', 'fsfpay-payment-gateway' ),
                    'type'    => 'checkbox',
                    'label'   => __( 'Enable FSFPAY Payment Gateway', 'fsfpay-payment-gateway' ),
                    'default' => 'yes',
                ),
                'title' => array(
                    'title'       => __( 'Title', 'fsfpay-payment-gateway' ),
                    'type'        => 'text',
                    'description' => __( 'Payment method title the customer will see at checkout.', 'fsfpay-payment-gateway' ),
                    'default'     => __( 'FSFPAY - Credit / Debit Card', 'fsfpay-payment-gateway' ),
                ),
                'description' => array(
                    'title'       => __( 'Description', 'fsfpay-payment-gateway' ),
                    'type'        => 'textarea',
                    'description' => __( 'Shown on checkout page under the gateway title.', 'fsfpay-payment-gateway' ),
                    'default'     => __( 'You will be redirected to secure payment page.', 'fsfpay-payment-gateway' ),
                ),
                'test_mode' => array(
                    'title'       => __( 'Test mode', 'fsfpay-payment-gateway' ),
                    'type'        => 'checkbox',
                    'label'       => __( 'Enable Test Mode (sandbox)', 'fsfpay-payment-gateway' ),
                    'default'     => 'yes',
                ),
                'api_key' => array(
                    'title'       => __( 'API Key', 'fsfpay-payment-gateway' ),
                    'type'        => 'text',
                    'description' => __( 'Your FSFPAY public API key', 'fsfpay-payment-gateway' ),
                    'default'     => '',
                ),
                'api_secret' => array(
                    'title'       => __( 'API Secret', 'fsfpay-payment-gateway' ),
                    'type'        => 'password',
                    'description' => __( 'Your FSFPAY secret key', 'fsfpay-payment-gateway' ),
                    'default'     => '',
                ),
                'test_endpoint' => array(
                    'title'       => __( 'Test Endpoint', 'fsfpay-payment-gateway' ),
                    'type'        => 'text',
                    'default'     => 'https://panel.fsfpay.com/api/sandbox/checkout',
                ),
                'live_endpoint' => array(
                    'title'       => __( 'Live Endpoint', 'fsfpay-payment-gateway' ),
                    'type'        => 'text',
                    'default'     => 'https://panel.fsfpay.com/api/checkout',
                ),
            );
        }

        public function admin_options() {
            echo '<h2>' . esc_html( $this->get_method_title() ) . '</h2>';
            echo wp_kses_post( wpautop( $this->get_method_description() ) );
            parent::admin_options();
        }

        public function payment_fields() {
            if ( $this->description ) {
                echo wpautop( wp_kses_post( $this->description ) );
            }
        }

        public function process_payment( $order_id ) {
            $order = wc_get_order( $order_id );

            $payload = array(
                'amount'       => round( $order->get_total(), 2 ),
                'currency'     => $order->get_currency(),
                'order_id'     => $order->get_id(),
                'customer'     => array(
                    'email' => $order->get_billing_email(),
                    'name'  => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                ),
                'return_url'   => $this->get_return_url( $order ),
                'notify_url'   => rest_url( '/fsfpay/v1/ipn' ),
            );

            $endpoint = $this->test_mode ? $this->test_endpoint : $this->live_endpoint;

            $response = $this->api_request( $endpoint . '/create-session', $payload );

            if ( is_wp_error( $response ) ) {
                $this->logger->error( 'FSFPAY create-session error: ' . $response->get_error_message(), array( 'source' => $this->log_channel ) );
                wc_add_notice( __( 'Payment error. Please try another payment method.', 'fsfpay-payment-gateway' ), 'error' );
                return array( 'result' => 'failure' );
            }

            $body = json_decode( $response['body'], true );

            if ( empty( $body ) || empty( $body['success'] ) || empty( $body['checkout_url'] ) ) {
                $this->logger->error( 'FSFPAY invalid response: ' . $response['body'], array( 'source' => $this->log_channel ) );
                wc_add_notice( __( 'Payment provider returned an unexpected response. Try again later.', 'fsfpay-payment-gateway' ), 'error' );
                return array( 'result' => 'failure' );
            }

            $order->update_status( 'on-hold', __( 'Awaiting FSFPAY payment', 'fsfpay-payment-gateway' ) );
            wc_reduce_stock_levels( $order_id );
            WC()->cart->empty_cart();

            return array(
                'result'   => 'success',
                'redirect' => esc_url_raw( $body['checkout_url'] ),
            );
        }

        protected function api_request( $url, $data = array() ) {
            $args = array(
                'timeout'   => 30,
                'headers'   => array(
                    'Content-Type'  => 'application/json',
                    'Accept'        => 'application/json',
                    'X-API-KEY'     => $this->api_key,
                    'X-API-SECRET'  => $this->api_secret,
                ),
                'body'      => wp_json_encode( $data ),
                'method'    => 'POST',
                'data_format' => 'body',
            );

            $this->logger->info( 'FSFPAY request to ' . $url . ' payload: ' . wp_json_encode( $data ), array( 'source' => $this->log_channel ) );

            $response = wp_remote_post( $url, $args );

            if ( is_wp_error( $response ) ) {
                return $response;
            }

            $code = wp_remote_retrieve_response_code( $response );
            $body = wp_remote_retrieve_body( $response );

            $this->logger->info( "FSFPAY response ({$code}): " . $body, array( 'source' => $this->log_channel ) );

            return array( 'code' => $code, 'body' => $body );
        }

        public function receipt_page( $order ) {
            echo '<p>' . __( 'You will be redirected to FSFPAY to complete your payment.', 'fsfpay-payment-gateway' ) . '</p>';
        }

        public function register_webhook_route() {
            register_rest_route( 'fsfpay/v1', '/ipn', array(
                'methods'  => 'POST',
                'callback' => array( $this, 'handle_ipn' ),
                'permission_callback' => '__return_true',
            ) );
        }

        public function handle_ipn( WP_REST_Request $request ) {
            $body = $request->get_body();
            $data = json_decode( $body, true );

            $this->logger->info( 'FSFPAY IPN received: ' . $body, array( 'source' => $this->log_channel ) );

            if ( empty( $data ) || empty( $data['order_id'] ) ) {
                return new WP_REST_Response( array( 'error' => 'invalid payload' ), 400 );
            }

            $order_id = intval( $data['order_id'] );
            $order = wc_get_order( $order_id );
            if ( ! $order ) {
                return new WP_REST_Response( array( 'error' => 'order not found' ), 404 );
            }

            if ( ! empty( $data['signature'] ) ) {
                $sig = $data['signature'];
                $computed = hash_hmac( 'sha256', $body, $this->api_secret );
                if ( ! hash_equals( $computed, $sig ) ) {
                    $this->logger->warning( 'FSFPAY IPN signature mismatch for order ' . $order_id, array( 'source' => $this->log_channel ) );
                    return new WP_REST_Response( array( 'error' => 'signature mismatch' ), 403 );
                }
            }

            $status = isset( $data['status'] ) ? $data['status'] : '';

            if ( in_array( $status, array( 'paid', 'completed' ), true ) ) {
                if ( $order->get_status() !== 'completed' ) {
                    $order->payment_complete( isset( $data['transaction_id'] ) ? $data['transaction_id'] : '' );
                    $order->add_order_note( 'FSFPAY payment completed via webhook.' );
                }
            } elseif ( in_array( $status, array( 'failed', 'cancelled', 'refused' ), true ) ) {
                $order->update_status( 'failed', 'FSFPAY reported payment failed/cancelled.' );
            } else {
                $order->add_order_note( 'FSFPAY webhook: status=' . $status );
            }

            return new WP_REST_Response( array( 'success' => true ), 200 );
        }

    }

    add_filter( 'woocommerce_payment_gateways', 'fsfpay_add_gateway_class' );
    function fsfpay_add_gateway_class( $methods ) {
        $methods[] = 'WC_Gateway_Fsfpay';
        return $methods;
    }
}
