<?php
namespace App\Http\Controllers\Fsfpay;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    // Show a simple checkout form (amount, currency, name, email)
    public function checkout()
    {
        return view('payments.iframe');
    }

    // Prepare payment and return iframe parameters (example)
    public function create(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.1',
            'name' => 'required|string',
            'email' => 'required|email',
            'lang' => 'nullable|string'
        ]);

        // Load credentials from env
        $username = env('FSFPAY_API_USERNAME');
        $apiKey = env('FSFPAY_API_KEY');
        $secret = env('FSFPAY_SECRET_KEY');
        $iframeUrl = env('FSFPAY_IFRAME_URL', 'https://panel.fsfpay.com/pay/iframe');

        // Build payload for FSFPAY API - this is an example shape.
        $payload = [
            'api_username' => $username,
            'api_key' => $apiKey,
            'amount' => number_format($validated['amount'], 2, '.', ''),
            'name' => $validated['name'],
            'email' => $validated['email'],
            'lang' => $validated['lang'] ?? app()->getLocale(),
            'callback_url' => route('fsfpay.callback', [], true) ?? ''
        ];

        // Example signature HMAC using secret key
        $signature = hash_hmac('sha256', json_encode($payload, JSON_UNESCAPED_SLASHES), $secret);
        $payload['signature'] = $signature;

        // In a real implementation you would POST $payload to FSFPAY server here,
        // get back a token/order id and then return the iframe URL with token.
        // For safety, we return a simulated iframe src with payload in query (NOT for production).

        $token = base64_encode(json_encode([
            'payload' => $payload,
            'ts' => time()
        ]));

        return response()->json([
            'iframe_src' => $iframeUrl . '?token=' . urlencode($token)
        ]);
    }
}
