<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class FsfpayController extends Controller
{
    // Show an iframe page for payment
    public function iframe(Request $request)
    {
        $orderId = $request->query('orderId', uniqid('order_'));
        $amount = $request->query('amount', '1.00');
        $currency = $request->query('currency', 'USD');
        $email = $request->query('email', 'buyer@example.com');
        return view('fsfpay.iframe', compact('orderId','amount','currency','email'));
    }

    // Create payment via FSFPAY API and return iframe src or redirect URL
    public function createPayment(Request $request)
    {
        $apiKey = env('FSFPAY_API_KEY', '');
        $apiUsername = env('FSFPAY_API_USERNAME', '');
        $secretKey = env('FSFPAY_SECRET_KEY', '');
        $endpoint = 'https://panel.fsfpay.com/api/payment';

        $data = [
            'orderId' => $request->input('orderId', uniqid('order_')),
            'email' => $request->input('email', 'buyer@example.com'),
            'amount' => $request->input('amount', '1.00'),
            'currency' => $request->input('currency', 'USD'),
            'cryptocurrency' => $request->input('cryptocurrency', 'BTC'),
            'testmode' => $request->input('testmode', 0),
            'callbackUrl' => $request->input('callbackUrl', url('/fsfpay/callback')),
            'successRedirectUrl' => $request->input('successRedirectUrl', url('/')),
            'language' => app()->getLocale(),
            'apiKey' => $apiKey,
            'apiUsername' => $apiUsername,
        ];
        $data['requestHash'] = hash('sha256', $apiKey . $apiUsername . $secretKey);

        // Use basic CURL via Http::withOptions if available, else fallback
        try {
            $response = Http::asForm()->post($endpoint, $data)->body();
        } catch (\Exception $e) {
            Log::error('FSFPAY createPayment error: '.$e->getMessage());
            return response()->json(['success'=>false,'message'=>'Request failed'],500);
        }

        $json = json_decode($response, true);
        if (!$json) {
            return response()->json(['success'=>false,'message'=>'Invalid JSON response'],500);
        }
        return response()->json($json);
    }

    // Handle callback from FSFPAY
    public function callback(Request $request)
    {
        $data = $request->all();
        Log::info('FSFPAY Callback:', $data);

        if (!isset($data['requestHash']) || !isset($data['paymentStatus'])) {
            return response(__('fsfpay.missing_parameters'), 400);
        }

        $apiKey = env('FSFPAY_API_KEY', '');
        $apiUsername = env('FSFPAY_API_USERNAME', '');
        $secretKey = env('FSFPAY_SECRET_KEY', '');

        $generatedHash = hash('sha256', $apiKey . $apiUsername . $secretKey);

        if ($data['requestHash'] === $generatedHash && $data['paymentStatus'] === 'successful') {
            $orderId = $data['orderId'] ?? null;
            $amount = $data['amount'] ?? null;
            $email = $data['email'] ?? null;
            $currency = $data['currency'] ?? null;

            // TODO: update your order / donation / record here
            // Example: \App\Models\Order::where('id',$orderId)->update(['status'=>'paid']);

            return response('OK', 200);
        }

        return response(__('fsfpay.invalid_hash_or_status'), 403);
    }

    // Get balances via FSFPAY API
    public function balances(Request $request)
    {
        $apiKey = env('FSFPAY_API_KEY', '');
        $apiUsername = env('FSFPAY_API_USERNAME', '');
        $secretKey = env('FSFPAY_SECRET_KEY', '');
        $endpoint = 'https://panel.fsfpay.com/api/balances';

        $data = ['apiKey' => $apiKey, 'apiUsername' => $apiUsername];
        $data['requestHash'] = hash('sha256', $apiKey . $apiUsername . $secretKey);

        try {
            $response = Http::asForm()->post($endpoint, $data)->body();
        } catch (\Exception $e) {
            Log::error('FSFPAY balances error: '.$e->getMessage());
            return response()->json(['success'=>false,'message'=>'Request failed'],500);
        }

        $json = json_decode($response, true);
        return response()->json($json);
    }
}
