/**
 * FSFPAY - Simple Shopify Custom App skeleton (Express)
 */
const express = require('express');
const bodyParser = require('body-parser');
const axios = require('axios');
const crypto = require('crypto');
require('dotenv').config();

const app = express();
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));
app.use(express.static('public'));

const PORT = process.env.PORT || 3000;

function getEndpoint() {
  const custom = process.env.FSFPAY_CUSTOM_ENDPOINT && process.env.FSFPAY_CUSTOM_ENDPOINT.trim();
  const testMode = process.env.FSFPAY_TEST_MODE === '1' || process.env.FSFPAY_TEST_MODE === 'true';
  if (custom) return custom.replace(/\/$/, '');
  return testMode ? (process.env.FSFPAY_TEST_ENDPOINT || 'https://panel.fsfpay.com/api/sandbox/checkout') : (process.env.FSFPAY_LIVE_ENDPOINT || 'https://panel.fsfpay.com/api/checkout');
}

app.post('/create-session', async (req, res) => {
  try {
    const endpoint = getEndpoint();
    const apiKey = process.env.FSFPAY_API_KEY || '';
    const apiSecret = process.env.FSFPAY_API_SECRET || '';

    const payload = {
      amount: parseFloat(req.body.amount || '0').toFixed(2),
      currency: req.body.currency || 'USD',
      order_id: req.body.order_id || Date.now().toString(),
      customer: { email: req.body.email || '', name: req.body.name || '' },
      return_url: req.body.return_url || (req.protocol + '://' + req.get('host') + '/return'),
      notify_url: req.body.notify_url || (req.protocol + '://' + req.get('host') + '/webhook')
    };

    const response = await axios.post(endpoint + '/create-session', payload, {
      headers: { 'Content-Type': 'application/json', 'Accept': 'application/json', 'X-API-KEY': apiKey, 'X-API-SECRET': apiSecret },
      timeout: 30000
    });

    if (response.data && response.data.success && response.data.checkout_url) {
      return res.json({ success: true, checkout_url: response.data.checkout_url });
    } else {
      console.error('FSFPAY create-session invalid response', response.data);
      return res.status(502).json({ success: false, error: 'Invalid response from payment provider', raw: response.data });
    }
  } catch (err) {
    console.error('FSFPAY create-session error', err.message || err);
    return res.status(500).json({ success: false, error: 'Server error' });
  }
});

app.post('/webhook', (req, res) => {
  const body = JSON.stringify(req.body);
  console.log('FSFPAY webhook received:', body);

  const signature = req.body.signature || req.get('x-fsfpay-signature') || '';
  const apiSecret = process.env.FSFPAY_API_SECRET || '';

  if (signature && apiSecret) {
    const computed = crypto.createHmac('sha256', apiSecret).update(body).digest('hex');
    if (computed !== signature) {
      console.warn('FSFPAY signature mismatch', computed, signature);
      return res.status(403).json({ error: 'signature mismatch' });
    }
  }

  const orderId = req.body.order_id;
  const status = req.body.status;
  console.log('Order', orderId, 'status', status);

  return res.json({ success: true });
});

app.get('/return', (req, res) => {
  res.send('<h2>Payment completed. Return page</h2><p>Close this window or return to Shopify.</p>');
});

app.get('/', (req, res) => {
  res.sendFile(__dirname + '/public/index.html');
});

app.listen(PORT, () => {
  console.log('FSFPAY Shopify App skeleton running on port', PORT);
});
