        <?php
        /**
         * Plugin Name: FSFPAY Shortcode
         * Description: Simple FSFPAY payment shortcode + admin settings. Modern responsive form, REST API integration, and IPN callback.
         * Version: 2.0
         * Author: Generated by FSFPAY
         * Text Domain: fsfpay
         * Domain Path: /languages
         */

        if ( ! defined( 'ABSPATH' ) ) {
            exit;
        }

        class FSFPAY_Shortcode_Plugin {

            private static $instance = null;
            private $option_group = 'fsfpay_options';
            private $option_name = 'fsfpay_settings';

            public static function instance() {
                if ( self::$instance === null ) {
                    self::$instance = new self();
                }
                return self::$instance;
            }

            private function __construct() {
                add_action('init', array($this, 'load_textdomain'));
                add_action('admin_menu', array($this, 'admin_menu'));
                add_action('admin_init', array($this, 'register_settings'));
                add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
                add_shortcode('fsfpay', array($this, 'shortcode_render'));
                add_action('rest_api_init', array($this, 'register_rest_routes'));
                add_action('admin_notices', array($this, 'maybe_admin_notice'));
            }

            public function load_textdomain() {
                load_plugin_textdomain('fsfpay', false, dirname(plugin_basename(__FILE__)) . '/languages');
            }

            public function admin_menu() {
                add_options_page(__('FSFPAY Settings','fsfpay'), __('FSFPAY','fsfpay'), 'manage_options', 'fsfpay-settings', array($this,'settings_page'));
            }

            public function register_settings() {
                register_setting($this->option_group, $this->option_name, array($this,'sanitize_settings'));
                add_settings_section('fsfpay_main', __('FSFPAY Configuration','fsfpay'), null, 'fsfpay-settings');

                add_settings_field('username', __('API Username','fsfpay'), array($this,'render_username'), 'fsfpay-settings', 'fsfpay_main');
                add_settings_field('apikey', __('API Key','fsfpay'), array($this,'render_apikey'), 'fsfpay-settings', 'fsfpay_main');
                add_settings_field('secret', __('API Secret','fsfpay'), array($this,'render_secret'), 'fsfpay-settings', 'fsfpay_main');
                add_settings_field('testmode', __('Test Mode','fsfpay'), array($this,'render_testmode'), 'fsfpay-settings', 'fsfpay_main');
                add_settings_field('custom_endpoint', __('Custom API Endpoint (optional)','fsfpay'), array($this,'render_custom_endpoint'), 'fsfpay-settings', 'fsfpay_main');
            }

            public function sanitize_settings($input) {
                $out = array();
                $out['username'] = sanitize_text_field($input['username'] ?? '');
                $out['apikey'] = sanitize_text_field($input['apikey'] ?? '');
                $out['secret'] = sanitize_text_field($input['secret'] ?? '');
                $out['testmode'] = isset($input['testmode']) ? 1 : 0;
                $out['custom_endpoint'] = esc_url_raw($input['custom_endpoint'] ?? '');
                return $out;
            }

            public function get_options() {
                $defaults = array(
                    'username' => '',
                    'apikey' => '',
                    'secret' => '',
                    'testmode' => 1,
                    'custom_endpoint' => '',
                );
                $opts = get_option($this->option_name, array());
                return wp_parse_args($opts, $defaults);
            }

            public function render_username() {
                $o = $this->get_options();
                printf('<input type="text" name="%s[username]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr($o['username']));
            }
            public function render_apikey() {
                $o = $this->get_options();
                printf('<input type="text" name="%s[apikey]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr($o['apikey']));
            }
            public function render_secret() {
                $o = $this->get_options();
                printf('<input type="password" autocomplete="new-password" name="%s[secret]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr($o['secret']));
            }
            public function render_testmode() {
                $o = $this->get_options();
                printf('<label><input type="checkbox" name="%s[testmode]" value="1" %s /> %s</label>', esc_attr($this->option_name), checked(1, $o['testmode'], false), __('Enable sandbox/test mode','fsfpay'));
            }
            public function render_custom_endpoint() {
                $o = $this->get_options();
                printf('<input type="text" name="%s[custom_endpoint]" value="%s" class="regular-text" placeholder="https://..." />', esc_attr($this->option_name), esc_attr($o['custom_endpoint']));
            }

            public function settings_page() {
                if (!current_user_can('manage_options')) return;
                $opts = $this->get_options();
                $webhook = rest_url('fsfpay/v1/ipn');
                ?>
                <div class="wrap">
                    <h1><?php _e('FSFPAY Settings','fsfpay'); ?></h1>
                    <form method="post" action="options.php">
                    <?php
                        settings_fields($this->option_group);
                        do_settings_sections('fsfpay-settings');
                        submit_button();
                    ?>
                    </form>

                    <h2><?php _e('Webhook / IPN URL','fsfpay'); ?></h2>
                    <p><?php _e('Set this Webhook URL in your FSFPAY panel so payment notifications can be delivered to your site.','fsfpay'); ?></p>
                    <pre style="background:#fff;padding:8px;border:1px solid #eee;"><?php echo esc_html($webhook); ?></pre>

                    <h2><?php _e('How to use the shortcode','fsfpay'); ?></h2>
                    <div style="background:#f7f7f7;padding:12px;border-left:4px solid #00a0d2;">
<pre style="white-space:pre-wrap;"><?php _e('Add a payment form to any page or post using this shortcode:', 'fsfpay'); ?>

[fsfpay amount="100" currency="EUR" description="Donation for Support"]

<?php _e('Parameters:', 'fsfpay'); ?>
• <?php _e('amount — Payment amount (required)', 'fsfpay'); ?>
• <?php _e('currency — Currency code (default EUR)', 'fsfpay'); ?>
• <?php _e('description — Payment description (optional)', 'fsfpay'); ?>

<?php _e('Example:', 'fsfpay'); ?>
[fsfpay amount="50" currency="USD" description="Membership Fee"]
</pre>
                    </div>

                </div>
                <?php
            }

            public function enqueue_assets() {
                wp_enqueue_style('fsfpay-style', plugin_dir_url(__FILE__) . 'assets/fsfpay.css', array(), '1.0');
                wp_enqueue_script('fsfpay-js', plugin_dir_url(__FILE__) . 'assets/fsfpay.js', array('jquery'), '1.0', true);
                wp_localize_script('fsfpay-js', 'fsfpay_vars', array(
                    'rest_url' => esc_url_raw(rest_url('fsfpay/v1/create')),
                    'nonce' => wp_create_nonce('wp_rest'),
                ));
            }

            public function shortcode_render($atts) {
                $atts = shortcode_atts(array(
                    'amount' => '',
                    'currency' => 'TRY',
                    'description' => '',
                ), $atts, 'fsfpay');

                $amount = esc_attr($atts['amount']);
                $currency = esc_attr($atts['currency']);
                $description = esc_attr($atts['description']);

                ob_start(); ?>
                <div class="fsfpay-form-wrap">
                    <form class="fsfpay-form" data-amount="<?php echo $amount; ?>" data-currency="<?php echo $currency; ?>">
                        <div class="fsfpay-row">
                            <label><?php _e('Amount','fsfpay'); ?></label>
                            <input type="number" name="amount" value="<?php echo $amount; ?>" required min="0.01" step="0.01" />
                        </div>
                        <div class="fsfpay-row">
                            <label><?php _e('Currency','fsfpay'); ?></label>
                            <input type="text" name="currency" value="<?php echo $currency; ?>" required />
                        </div>
                        <div class="fsfpay-row">
                            <label><?php _e('Description','fsfpay'); ?></label>
                            <input type="text" name="description" value="<?php echo $description; ?>" />
                        </div>
                        <div class="fsfpay-row">
                            <button type="submit" class="fsfpay-btn"><?php _e('Pay with FSFPAY','fsfpay'); ?></button>
                        </div>
                        <div class="fsfpay-row fsfpay-note">
                            <small><?php _e('Payments are securely processed by FSFPAY.','fsfpay'); ?></small>
                        </div>
                    </form>
                </div>
                <?php
                return ob_get_clean();
            }

            public function register_rest_routes() {
                register_rest_route('fsfpay/v1', '/create', array(
                    'methods' => 'POST',
                    'callback' => array($this, 'rest_create'),
                    'permission_callback' => '__return_true',
                ));
                register_rest_route('fsfpay/v1', '/ipn', array(
                    'methods' => 'POST',
                    'callback' => array($this, 'rest_ipn'),
                    'permission_callback' => '__return_true',
                ));
            }

            public function rest_create($request) {
                $params = $request->get_json_params();
                // simple nonce check
                $headers = getallheaders();
                if (empty($headers['X-WP-Nonce']) && empty($params['nonce'])) {
                    return rest_ensure_response(array('success' => false, 'message' => __('Missing nonce','fsfpay')));
                }
                // validate input
                $amount = isset($params['amount']) ? sanitize_text_field($params['amount']) : '';
                if (!$amount || !is_numeric($amount)) {
                    return rest_ensure_response(array('success' => false, 'message' => __('Invalid amount','fsfpay')));
                }
                $description = sanitize_text_field($params['description'] ?? '');
                $currency = sanitize_text_field($params['currency'] ?? 'TRY');

                $opts = $this->get_options();
                if (empty($opts['username']) || empty($opts['apikey']) || empty($opts['secret'])) {
                    return rest_ensure_response(array('success' => false, 'message' => __('API credentials not configured','fsfpay')));
                }

                // choose endpoint
                $endpoint = $opts['custom_endpoint'];
                if (empty($endpoint)) {
                    if ($opts['testmode']) {
                        $endpoint = 'https://panel.fsfpay.com/api/sandbox/checkout';
                    } else {
                        $endpoint = 'https://panel.fsfpay.com/api/checkout';
                    }
                }

                $payload = array(
                    'username' => $opts['username'],
                    'apikey' => $opts['apikey'],
                    'secret' => $opts['secret'],
                    'amount' => $amount,
                    'currency' => $currency,
                    'description' => $description,
                    'callback' => rest_url('fsfpay/v1/ipn'),
                );

                $args = array(
                    'body' => wp_json_encode($payload),
                    'headers' => array(
                        'Content-Type' => 'application/json'
                    ),
                    'timeout' => 20,
                );

                $response = wp_remote_post($endpoint, $args);
                if (is_wp_error($response)) {
                    return rest_ensure_response(array('success' => false, 'message' => $response->get_error_message()));
                }

                $code = wp_remote_retrieve_response_code($response);
                $body = wp_remote_retrieve_body($response);
                $json = json_decode($body, true);

                if ($code !== 200 || !is_array($json)) {
                    return rest_ensure_response(array('success' => false, 'message' => __('Payment provider returned an unexpected response.','fsfpay'), 'raw' => $body));
                }

                // expected provider response: success + redirect url or payment id; adapt if needed
                if (!empty($json['success']) && $json['success'] === true && !empty($json['data']['redirect'])) {
                    return rest_ensure_response(array('success' => true, 'redirect' => esc_url_raw($json['data']['redirect'])));
                }

                // fallback: return entire provider response
                return rest_ensure_response(array('success' => false, 'message' => __('Payment provider returned an unexpected response.','fsfpay'), 'raw' => $body, 'data' => $json));
            }

            public function rest_ipn($request) {
                $params = $request->get_json_params();
                if (!$params) {
                    $params = $_POST;
                }
                $opts = $this->get_options();
                // simple secret validation if provider sends secret back
                if (!empty($opts['secret']) && isset($params['secret']) && $params['secret'] !== $opts['secret']) {
                    return new WP_REST_Response(array('success' => false, 'message' => 'Invalid secret'), 403);
                }

                // handle ipn: developer should customize this to update orders, send emails, etc.
                // We'll log the IPN for admins
                if (function_exists('error_log')) {
                    error_log('[fsfpay ipn] ' . print_r($params, true));
                }

                return new WP_REST_Response(array('success' => true), 200);
            }

            public function maybe_admin_notice() {
                $screen = get_current_screen();
                if ($screen && $screen->id === 'settings_page_fsfpay-settings') return;
                $opts = $this->get_options();
                if (empty($opts['apikey']) || empty($opts['secret']) || empty($opts['username'])) {
                    echo '<div class="notice notice-warning is-dismissible"><p>' . __('FSFPAY: API credentials not configured. Go to Settings → FSFPAY to set them.','fsfpay') . '</p></div>';
                }
            }

        }

        FSFPAY_Shortcode_Plugin::instance();

