<?php
/*
 * Plugin Name: Fsfpay Crypto Payment Gateway
 * Plugin URI: https://fsfpay.com/
 * Description: Accept cryptocurrency payments securely with Fsfpay. Works with all WooCommerce themes.
 * Author: Fsfpay
 * Author URI: https://fsfpay.com/
 * Version: 1.0.9
 * Text Domain: fsfpay-payment
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) exit;

add_filter('woocommerce_payment_gateways', 'fsfpay_add_gateway_class');
function fsfpay_add_gateway_class($gateways) {
    $gateways[] = 'WC_Gateway_Fsfpay';
    return $gateways;
}

add_action('plugins_loaded', 'fsfpay_init_gateway_class');
function fsfpay_init_gateway_class() {
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_Gateway_Fsfpay extends WC_Payment_Gateway {
        public string $base_url = 'https://panel.fsfpay.com/';

        public function __construct() {
            $this->id = 'fsfpay';
            $this->has_fields = false;
            $this->method_title = __('Fsfpay (Crypto)', 'fsfpay-payment');
            $this->method_description = __('Pay securely using cryptocurrency via Fsfpay API.', 'fsfpay-payment');
            $this->supports = ['products'];

            $this->init_form_fields();
            $this->init_settings();

            $this->title        = $this->get_option('title');
            $this->description  = $this->get_option('description');
            $this->enabled      = $this->get_option('enabled');
            $this->api_username = $this->get_option('api_username');
            $this->api_key      = $this->get_option('api_key');
            $this->secret_key   = $this->get_option('secret_key');
            $this->order_status = $this->get_option('order_status');
            $this->test_mode    = $this->get_option('test_mode');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
            add_action('woocommerce_receipt_' . $this->id, [$this, 'receipt_page']);
            add_action('woocommerce_api_' . $this->id, [$this, 'callback_handler']);
        }

        public function is_available() {
            if ('yes' !== $this->enabled) return false;
            if (empty($this->api_key) || empty($this->api_username) || empty($this->secret_key)) return false;
            if (is_admin() && !defined('DOING_AJAX')) return false;
            return parent::is_available();
        }

public function admin_options() {
    echo '<h2>' . esc_html($this->get_method_title()) . '</h2>';
    echo wp_kses_post('<p>' . $this->get_method_description() . '</p>');
    echo '<table class="form-table">';
    $this->generate_settings_html();
    echo '</table>';

    // 🧩 Troubleshooting info box
    echo '<hr>';
    echo '<div style="background:#f9f9f9;border:1px solid #ccd0d4;padding:15px 20px;border-radius:8px;margin-top:20px;">';
    echo '<h3 style="margin-top:0;">🛠️ Troubleshooting</h3>';
    echo '<p><strong>Checkout Method Compatibility</strong></p>';
    echo '<p>WooCommerce has two checkout types: <strong>Block Checkout</strong> and <strong>Classic Checkout</strong>.</p>';
    echo '<p><strong>FSFPAY currently supports only the Classic Checkout method.</strong></p>';

    echo '<p>If you see the following message on your checkout page:</p>';
    echo '<blockquote style="margin:10px 0;padding:10px;background:#fff;border-left:4px solid #0073aa;">“There are no payment methods available.”</blockquote>';

    echo '<p>To fix this, follow these steps:</p>';
    echo '<ol style="margin-left:20px;">';
    echo '<li>In your WordPress admin panel, go to <strong>Pages → Checkout</strong>.</li>';
    echo '<li>Edit the page.</li>';
    echo '<li>Click the <strong>“Switch to classic checkout”</strong> button shown in the editor sidebar — this will automatically convert your checkout page to the Classic Checkout template.</li>';
    echo '<li>Save the page and refresh your checkout.</li>';
    echo '</ol>';

    echo '<p>After switching, the FSFPAY payment option will appear and work correctly.</p>';

    echo '<p><strong>Alternatively (manual method):</strong></p>';
    echo '<p>If you are using the Classic Editor, replace the page content with this shortcode:</p>';
    echo '<pre style="background:#eee;padding:10px;border-radius:5px;">[woocommerce_checkout]</pre>';
    echo '</div>';
}



        public function enqueue_scripts() {
            if (is_checkout() || is_page('checkout') || isset($_GET['order-pay'])) {
                wp_enqueue_style('fsfpay-style', plugins_url('assets/css/fsfpay15.css', __FILE__));
                wp_enqueue_script('fsfpay-js', plugins_url('assets/js/fsfpay-widget14.js', __FILE__), ['jquery'], null, true);
            }
        }

        public function init_form_fields() {
            $this->form_fields = [
                'enabled' => [
                    'title' => __('Enable/Disable', 'fsfpay-payment'),
                    'type' => 'checkbox',
                    'label' => __('Enable Fsfpay Payment', 'fsfpay-payment'),
                    'default' => 'no'
                ],
                'title' => [
                    'title' => __('Title', 'fsfpay-payment'),
                    'type' => 'text',
                    'default' => __('Pay with Cryptocurrency', 'fsfpay-payment'),
                ],
                'description' => [
                    'title' => __('Description', 'fsfpay-payment'),
                    'type' => 'textarea',
                    'default' => __('You can make secure cryptocurrency payments.', 'fsfpay-payment'),
                ],
                'api_username' => ['title' => __('API Username', 'fsfpay-payment'), 'type' => 'text'],
                'api_key' => ['title' => __('API Key', 'fsfpay-payment'), 'type' => 'text'],
                'secret_key' => ['title' => __('Secret Key', 'fsfpay-payment'), 'type' => 'password'],
                'order_status' => [
                    'title' => __('Order Completed Status', 'fsfpay-payment'),
                    'type' => 'select',
                    'options' => ['processing'=>__('Processing','fsfpay-payment'),'completed'=>__('Completed','fsfpay-payment')],
                    'default' => 'processing'
                ],
                'test_mode' => [
                    'title' => __('Test Mode', 'fsfpay-payment'),
                    'type' => 'checkbox',
                    'label' => __('Enable Test Mode', 'fsfpay-payment'),
                    'default' => 'no'
                ],
            ];
        }

        public function payment_fields() {
            if ($this->description) echo wpautop(wptexturize($this->description));
            echo '<div class="fsfpay-widget">';
            echo '<p><strong>' . esc_html__('After placing your order, you will be redirected to choose your cryptocurrency.', 'fsfpay-payment') . '</strong></p>';
            echo '</div>';
        }

        public function process_payment($order_id) {
            $order = wc_get_order($order_id);
            return ['result'=>'success','redirect'=>$order->get_checkout_payment_url(true)];
        }

        public function receipt_page($order_id) {
            $order = wc_get_order($order_id);
            $order_data = $order->get_data();
            $currency = strtolower($order_data['currency']);
            $callback_url = get_site_url(null, '?wc-api=' . $this->id);

            if (!isset($_GET['coin'])) {
                $response = $this->fsfpay_request('cryptocurrencies');
                if (!empty($response['success'])) {
                    $cryptocurrencies = $response['cryptocurrencies'];
                    $gateway = $this;
                    include __DIR__ . '/includes/cryptocurrencies.php';
                } else {
                    echo '<div style="color:red;">Fsfpay Error: ' . esc_html($response['message'] ?? 'API connection failed') . '</div>';
                }
                return;
            }

            $coin = sanitize_text_field($_GET['coin']);
            $response = $this->fsfpay_request('payment', [
                'orderId'=>$order_id,
                'email'=>$order->get_billing_email(),
                'amount'=>floatval(number_format($order_data['total'],2,'.','')),
                'currency'=>strtoupper($currency),
                'cryptocurrency'=>strtoupper($coin),
                'testmode'=>$this->test_mode=='no'?0:1,
                'callbackUrl'=>$callback_url,
                'successRedirectUrl'=>$order->get_checkout_order_received_url()
            ]);

            if (!empty($response['success']) && !empty($response['scriptUrl'])) {
                echo '<div id="fsfpay-widget"></div>';
                echo '<script src="' . esc_url($response['scriptUrl']) . '"></script>';
            } else {
                echo '<div style="color:red;">Fsfpay Error: ' . esc_html($response['message'] ?? 'Unknown error') . '</div>';
            }

            echo '<a href="' . esc_url($order->get_checkout_payment_url()) . '" class="fsfpay-change-coin">'
                 . __('Choose another cryptocurrency','fsfpay-payment') . '</a>';
        }

        private function fsfpay_request($path,$req=[]) {
            $data = [
                'apiKey'=>$this->api_key,
                'apiUsername'=>$this->api_username,
                'requestHash'=>hash('sha256',$this->api_key.$this->api_username.$this->secret_key),
                'language'=>explode('_',get_locale())[0]
            ];
            $data = array_merge($data,$req);

            $ch=curl_init();
            curl_setopt_array($ch,[
                CURLOPT_URL=>$this->base_url.'api/'.$path,
                CURLOPT_HTTPHEADER=>['Content-Type: application/x-www-form-urlencoded'],
                CURLOPT_POST=>true,
                CURLOPT_RETURNTRANSFER=>true,
                CURLOPT_POSTFIELDS=>http_build_query($data)
            ]);
            $response=curl_exec($ch);
            curl_close($ch);
            return json_decode($response,true);
        }

        public function callback_handler() {
            if(!isset($_POST['requestHash']) || !isset($_POST['paymentStatus'])) die('Missing fields!');
            if($_POST['paymentStatus']!=='successful') die('Invalid status');

            $verify=hash('sha256',$this->api_key.$this->api_username.$this->secret_key);
            if($verify!==$_POST['requestHash']) die('Invalid hash!');

            $order=wc_get_order(intval($_POST['orderId']));
            if(!$order) die('Order not found');

            if(in_array($order->get_status(),['pending','failed'])){
                wc_reduce_stock_levels($order->get_id());
                $order->add_order_note(sprintf(
                    __('Fsfpay Payment ID: %s | Amount: %s %s','fsfpay-payment'),
                    sanitize_text_field($_POST['paymentId']),
                    sanitize_text_field($_POST['cryptocurrencyAmount']),
                    sanitize_text_field($_POST['cryptocurrency'])
                ));
                $order->update_status($this->order_status);
            }
            die('OK');
        }
    }
}