<?php
/*
Plugin Name: FSFPAY Checkout Shortcode
Plugin URI: https://panel.fsfpay.com/
Description: Add FSFPAY hosted checkout buttons inside posts/pages via shortcode. Admin settings for API keys and endpoints. Webhook listener included.
Version: 1.0.0
Author: FSFPAY Integration
Text Domain: fsfpay-wp
*/

if (!defined('ABSPATH')) {
    exit;
}

class FSFPAY_WP_Plugin {
    private static $instance = null;
    private $option_name = 'fsfpay_wp_options';

    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct() {
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('rest_api_init', array($this, 'register_routes'));
        add_shortcode('fsfpay_button', array($this, 'render_shortcode'));
    }

    public function admin_menu() {
        add_options_page('FSFPAY Settings', 'FSFPAY', 'manage_options', 'fsfpay-settings', array($this, 'settings_page'));
    }

    public function settings_init() {
        register_setting($this->option_name, $this->option_name, array($this, 'validate_options'));
        add_settings_section('fsfpay_main', 'FSFPAY Settings', null, 'fsfpay-settings');

        add_settings_field('api_key', 'API Key', array($this, 'field_api_key'), 'fsfpay-settings', 'fsfpay_main');
        add_settings_field('api_secret', 'API Secret', array($this, 'field_api_secret'), 'fsfpay-settings', 'fsfpay_main');
        add_settings_field('test_mode', 'Test Mode', array($this, 'field_test_mode'), 'fsfpay-settings', 'fsfpay_main');
        add_settings_field('test_endpoint', 'Test Endpoint', array($this, 'field_test_endpoint'), 'fsfpay-settings', 'fsfpay_main');
        add_settings_field('live_endpoint', 'Live Endpoint', array($this, 'field_live_endpoint'), 'fsfpay-settings', 'fsfpay_main');
        add_settings_field('custom_endpoint', 'Custom API Endpoint (optional)', array($this, 'field_custom_endpoint'), 'fsfpay-settings', 'fsfpay_main');
    }

    public function validate_options($input) {
        $out = array();
        $out['api_key'] = sanitize_text_field($input['api_key']);
        $out['api_secret'] = sanitize_text_field($input['api_secret']);
        $out['test_mode'] = isset($input['test_mode']) ? 1 : 0;
        $out['test_endpoint'] = esc_url_raw($input['test_endpoint']);
        $out['live_endpoint'] = esc_url_raw($input['live_endpoint']);
        $out['custom_endpoint'] = trim($input['custom_endpoint']);
        if ($out['custom_endpoint'] && stripos($out['custom_endpoint'], 'https://') !== 0) {
            add_settings_error($this->option_name, 'invalid_endpoint', 'Custom endpoint must start with https://', 'error');
            // keep previous saved value
            $saved = get_option($this->option_name, array());
            $out['custom_endpoint'] = isset($saved['custom_endpoint']) ? $saved['custom_endpoint'] : '';
        }
        return $out;
    }

    public function field_api_key() {
        $opts = get_option($this->option_name, array());
        printf('<input type="text" name="%s[api_key]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr(isset($opts['api_key']) ? $opts['api_key'] : ''));
    }
    public function field_api_secret() {
        $opts = get_option($this->option_name, array());
        printf('<input type="password" name="%s[api_secret]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr(isset($opts['api_secret']) ? $opts['api_secret'] : ''));
    }
    public function field_test_mode() {
        $opts = get_option($this->option_name, array());
        $checked = isset($opts['test_mode']) && $opts['test_mode'] ? 'checked' : '';
        printf('<input type="checkbox" name="%s[test_mode]" value="1" %s /> Enable sandbox/test mode', esc_attr($this->option_name), $checked);
    }
    public function field_test_endpoint() {
        $opts = get_option($this->option_name, array());
        $val = isset($opts['test_endpoint']) ? $opts['test_endpoint'] : 'https://panel.fsfpay.com/api/sandbox/checkout';
        printf('<input type="text" name="%s[test_endpoint]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr($val));
    }
    public function field_live_endpoint() {
        $opts = get_option($this->option_name, array());
        $val = isset($opts['live_endpoint']) ? $opts['live_endpoint'] : 'https://panel.fsfpay.com/api/checkout';
        printf('<input type="text" name="%s[live_endpoint]" value="%s" class="regular-text" />', esc_attr($this->option_name), esc_attr($val));
    }
    public function field_custom_endpoint() {
        $opts = get_option($this->option_name, array());
        $val = isset($opts['custom_endpoint']) ? $opts['custom_endpoint'] : '';
        printf('<input type="text" name="%s[custom_endpoint]" value="%s" class="regular-text" /> <p class="description">Optional. If set must start with https://</p>', esc_attr($this->option_name), esc_attr($val));
    }

    public function settings_page() {
        ?>
        <div class="wrap">
            <h1>FSFPAY Settings</h1>
            <form method="post" action="options.php">
            <?php
                settings_fields($this->option_name);
                do_settings_sections('fsfpay-settings');
                submit_button();
            ?>
            </form>
            <h2>Webhook URL</h2>
            <p>Set this in your FSFPAY panel: <code><?php echo esc_url(rest_url('fsfpay/v1/ipn')); ?></code></p>
        </div>
        <?php
    }

    public function register_routes() {
        register_rest_route('fsfpay/v1', '/ipn', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_ipn'),
            'permission_callback' => '__return_true',
        ));
    }

    public function handle_ipn($request) {
        $body = $request->get_body();
        $data = json_decode($body, true);
        error_log('FSFPAY IPN received: ' . $body);
        if (empty($data) || empty($data['order_id'])) {
            return new WP_REST_Response(array('error' => 'invalid payload'), 400);
        }
        $opts = get_option($this->option_name, array());
        if (!empty($data['signature']) && !empty($opts['api_secret'])) {
            $computed = hash_hmac('sha256', $body, $opts['api_secret']);
            if (!hash_equals($computed, $data['signature'])) {
                error_log('FSFPAY IPN signature mismatch for order ' . intval($data['order_id']));
                return new WP_REST_Response(array('error' => 'signature mismatch'), 403);
            }
        }
        $order_id = intval($data['order_id']);
        $status = isset($data['status']) ? $data['status'] : '';
        // No native WooCommerce order here — this is for blog posts.
        // We store a simple record in WP options as a log for site owner.
        $logs = get_option('fsfpay_wp_logs', array());
        $logs[] = array('time' => current_time('mysql'), 'order_id' => $order_id, 'status' => $status, 'payload' => $data);
        update_option('fsfpay_wp_logs', $logs);
        return new WP_REST_Response(array('success' => true), 200);
    }

    private function resolve_endpoint() {
        $opts = get_option($this->option_name, array());
        $custom = isset($opts['custom_endpoint']) ? trim($opts['custom_endpoint']) : '';
        $test = isset($opts['test_mode']) && $opts['test_mode'];
        if ($custom) return $custom;
        return $test ? (isset($opts['test_endpoint']) ? $opts['test_endpoint'] : 'https://panel.fsfpay.com/api/sandbox/checkout') : (isset($opts['live_endpoint']) ? $opts['live_endpoint'] : 'https://panel.fsfpay.com/api/checkout');
    }

    public function render_shortcode($atts = array()) {
        $atts = shortcode_atts(array(
            'amount' => '0.00',
            'currency' => 'USD',
            'description' => 'Payment',
            'label' => 'Pay Now',
            'order_id' => '', // optional merchant-generated id
        ), $atts, 'fsfpay_button');

        $amount = number_format(floatval($atts['amount']), 2, '.', '');
        $currency = strtoupper(sanitize_text_field($atts['currency']));
        $description = esc_attr($atts['description']);
        $order_id = sanitize_text_field($atts['order_id']);

        $endpoint = $this->resolve_endpoint();
        $opts = get_option($this->option_name, array());

        $payload = array(
            'amount' => $amount,
            'currency' => $currency,
            'order_id' => $order_id ? $order_id : time(),
            'customer' => array(
                'email' => '',
                'name' => ''
            ),
            'return_url' => esc_url(home_url('/')),
            'notify_url' => esc_url(rest_url('fsfpay/v1/ipn')),
            'description' => $description,
        );

        // We'll create a small local handler that requests create-session via AJAX endpoint on this site
        // to avoid exposing secret in frontend. Provide a nonce-protected AJAX action.
        wp_enqueue_script('fsfpay-wp-js', plugins_url('assets/fsfpay.js', __FILE__), array('jquery'), '1.0', true);
        wp_localize_script('fsfpay-wp-js', 'fsfpay_wp', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('fsfpay_wp_nonce'),
            'endpoint' => rtrim($endpoint, '/'),
            'api_key' => isset($opts['api_key']) ? $opts['api_key'] : '',
        ));

        $button = '<button class="fsfpay-wp-button" data-amount="'.esc_attr($amount).'" data-currency="'.esc_attr($currency).'" data-description="'.esc_attr($description).'" data-order="'.esc_attr($payload['order_id']).'">'.esc_html($atts['label']).'</button>';

        return '<div class="fsfpay-wp-wrapper">'.$button.'</div>';
    }

}

// AJAX handler to create session server-side
add_action('wp_ajax_nopriv_fsfpay_create_session', 'fsfpay_create_session');
add_action('wp_ajax_fsfpay_create_session', 'fsfpay_create_session');

function fsfpay_create_session() {
    check_ajax_referer('fsfpay_wp_nonce', 'nonce');

    $opts = get_option('fsfpay_wp_options', array());
    $api_key = isset($opts['api_key']) ? $opts['api_key'] : '';
    $api_secret = isset($opts['api_secret']) ? $opts['api_secret'] : '';
    $endpoint = isset($opts['custom_endpoint']) && trim($opts['custom_endpoint']) ? trim($opts['custom_endpoint']) : (isset($opts['test_mode']) && $opts['test_mode'] ? (isset($opts['test_endpoint']) ? $opts['test_endpoint'] : 'https://panel.fsfpay.com/api/sandbox/checkout') : (isset($opts['live_endpoint']) ? $opts['live_endpoint'] : 'https://panel.fsfpay.com/api/checkout'));

    $amount = isset($_POST['amount']) ? floatval($_POST['amount']) : 0;
    $currency = isset($_POST['currency']) ? sanitize_text_field($_POST['currency']) : 'USD';
    $description = isset($_POST['description']) ? sanitize_text_field($_POST['description']) : '';
    $order_id = isset($_POST['order_id']) ? sanitize_text_field($_POST['order_id']) : time();

    $payload = array(
        'amount' => number_format($amount, 2, '.', ''),
        'currency' => $currency,
        'order_id' => $order_id,
        'customer' => array('email' => '', 'name' => ''),
        'return_url' => esc_url(home_url('/')),
        'notify_url' => esc_url(rest_url('fsfpay/v1/ipn')),
        'description' => $description,
    );

    $ch = curl_init();
    curl_setopt_array($ch, array(
        CURLOPT_URL => rtrim($endpoint, '/') . '/create-session',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => array(
            'Content-Type: application/json',
            'Accept: application/json',
            'X-API-KEY: ' . $api_key,
            'X-API-SECRET: ' . $api_secret,
        ),
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_TIMEOUT => 30,
    ));
    $response = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($err) {
        wp_send_json_error(array('message' => 'Payment provider error: '.$err), 500);
    }

    $body = json_decode($response, true);
    if (empty($body) || empty($body['success']) || empty($body['checkout_url'])) {
        wp_send_json_error(array('message' => 'Payment provider returned an unexpected response.', 'raw' => $response), 500);
    }

    wp_send_json_success(array('checkout_url' => $body['checkout_url']));
}

FSFPAY_WP_Plugin::instance();

?>
